// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import CoreImage
import UIKit
import CoreMedia
import Combine
import MSSImageScanner

class SwiftUISampleViewModel: ObservableObject {
    @Published var frame: UIImage?
    @Published var error: Error?
    @Published var scanResult: ScanResult?

    private let cameraManager = CameraManager()
    private let context = CIContext()
    
    init() {
        cameraManager.delegate = self
    }
    
    // MARK: - Public methods
    func stopSession() {
        cameraManager.stopSession()
    }
    
    func startSession() {
        cameraManager.startSession()
    }
}

extension SwiftUISampleViewModel: CameraManagerDelegate {
    func cameraManager(didOutput sampleBuffer: CMSampleBuffer) {
        guard let uiImage = makeUIImage(from: sampleBuffer) else {
            return
        }
                        
        Task.detached(priority: .userInitiated) {
            self.decodeImage(uiImage)
        }
        
        Task { @MainActor in
            self.frame = uiImage
        }
    }
    
    func cameraManager(didFailWithError error: CameraManagerError) {
        Task { @MainActor in
            self.error = error
        }
    }
}

extension SwiftUISampleViewModel {
    /// Used to convert `CMSampleBuffer` data from `AVFoundation` to `UIImage` object
    private func makeUIImage(from sampleBuffer: CMSampleBuffer?) -> UIImage? {
        guard
            let sampleBuffer,
            let cvBuffer = CMSampleBufferGetImageBuffer(sampleBuffer) else {
            return nil
        }

        let ciImage = CIImage(cvPixelBuffer: cvBuffer)

        guard let cgImage = context.createCGImage(ciImage, from: ciImage.extent) else {
            return nil
        }

        return UIImage(cgImage: cgImage)
    }

    /// Actual `UIImage` decoding stage
    private func decodeImage(_ image: UIImage) {
        do {
            let output = try QRCodeScannerSDK.decodeImage(image, codeType: .all)
            Task { @MainActor in
                self.scanResult = .scanned(output.result, codeType: output.codeType)
            }
        } catch ScannerError.invalidImage {
            // Code not decoded - try in next frame
        } catch let error {
            Task { @MainActor in
                self.scanResult = .error(error)
            }
        }
    }
}
